<?php
/**
 * Admin interface for Image Compression Plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class ICP_Admin {
    
    private $tinypng_api;
    
    public function __construct() {
        $this->tinypng_api = new ICP_TinyPNG_API();
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Register settings
        add_action('admin_init', array($this, 'register_settings'));
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_global_media_assets'));
        
        // Enqueue scripts for media modal specifically
        add_action('wp_enqueue_media', array($this, 'enqueue_media_scripts'));
        
        // Add settings link to plugins page
        add_filter('plugin_action_links_' . plugin_basename(ICP_PLUGIN_PATH . 'image-compression-plugin.php'), array($this, 'add_settings_link'));
        
        // Add compression status column to media library
        add_filter('manage_media_columns', array($this, 'add_media_column'));
        add_action('manage_media_custom_column', array($this, 'display_media_column'), 10, 2);
        
        // Add compression button to attachment details modal
        add_filter('attachment_fields_to_edit', array($this, 'add_compression_field_to_modal'), 10, 2);
        
        // AJAX handlers
        add_action('wp_ajax_icp_test_api_key', array($this, 'ajax_test_api_key'));
        add_action('wp_ajax_icp_reset_stats', array($this, 'ajax_reset_stats'));
        add_action('wp_ajax_icp_cleanup_orphaned', array($this, 'ajax_cleanup_orphaned'));
        add_action('wp_ajax_icp_save_bulk_settings', array($this, 'ajax_save_bulk_settings'));
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'Image Compression',
            'Image Compression',
            'manage_options',
            'image-compression',
            array($this, 'dashboard_page'),
            'dashicons-format-image',
            25
        );
        
        add_submenu_page(
            'image-compression',
            'Dashboard',
            'Dashboard',
            'manage_options',
            'image-compression',
            array($this, 'dashboard_page')
        );
        
        add_submenu_page(
            'image-compression',
            'Bulk Compression',
            'Bulk Compression',
            'manage_options',
            'image-compression-bulk',
            array($this, 'bulk_compression_page')
        );
        
        add_submenu_page(
            'image-compression',
            'Scheduled Compression',
            'Scheduled Compression',
            'manage_options',
            'image-compression-scheduler',
            array($this, 'scheduler_page')
        );

        add_submenu_page(
            'image-compression',
            'Restore Images',
            'Restore Images',
            'manage_options',
            'image-compression-restore',
            array($this, 'restore_page')
        );

        add_submenu_page(
            'image-compression',
            'Settings',
            'Settings',
            'manage_options',
            'image-compression-settings',
            array($this, 'settings_page')
        );
    }
    
    public function register_settings() {
        register_setting('icp_settings_group', 'icp_settings', array($this, 'sanitize_settings'));
    }
    
    public function sanitize_settings($input) {
        $sanitized = array();
        
        if (isset($input['api_key'])) {
            $sanitized['api_key'] = sanitize_text_field($input['api_key']);
        }
        
        $sanitized['auto_compress'] = isset($input['auto_compress']) ? 1 : 0;
        
        // Test API key if it's changed
        if (isset($input['api_key']) && !empty($input['api_key'])) {
            $test_result = $this->tinypng_api->test_api_key($input['api_key']);
            if (!$test_result['success']) {
                add_settings_error(
                    'icp_settings',
                    'api_key_invalid',
                    'API Key Error: ' . $test_result['error'],
                    'error'
                );
            } else {
                add_settings_error(
                    'icp_settings',
                    'api_key_valid',
                    'API key is valid and working!',
                    'updated'
                );
            }
        }
        
        return $sanitized;
    }
    
    public function dashboard_page() {
        $total_savings = ICP_Database::get_total_savings();
        
        // Handle search and pagination parameters
        $search = isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '';
        $status_filter = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
        $date_from = isset($_GET['date_from']) ? sanitize_text_field($_GET['date_from']) : '';
        $date_to = isset($_GET['date_to']) ? sanitize_text_field($_GET['date_to']) : '';
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $per_page = isset($_GET['per_page']) ? max(5, min(50, intval($_GET['per_page']))) : 10;
        
        // Get paginated results
        $compression_data = ICP_Database::get_compression_stats_paginated(array(
            'per_page' => $per_page,
            'page' => $current_page,
            'search' => $search,
            'status' => $status_filter,
            'date_from' => $date_from,
            'date_to' => $date_to
        ));
        
        $recent_compressions = $compression_data['items'];
        $available_statuses = ICP_Database::get_available_statuses();
        
        ?>
        <div class="wrap">
            <h1>Image Compression Dashboard</h1>
            
            <div class="icp-dashboard-actions" style="margin-bottom: 20px;">
                <button id="icp-reset-stats" class="button button-secondary" onclick="confirmResetStats()">
                    Reset All Statistics
                </button>
                <button id="icp-cleanup-orphaned" class="button button-secondary" onclick="confirmCleanupOrphaned()">
                    Clean Up Deleted Images
                </button>
            </div>

            
            <!-- Summary Cards -->
            <div class="icp-dashboard-cards">
                <div class="icp-card">
                    <div class="icp-card-header">
                        <h3>Total Images Compressed</h3>
                    </div>
                    <div class="icp-card-body">
                        <div class="icp-stat-number"><?php echo number_format($total_savings->total_images); ?></div>
                    </div>
                </div>
                
                <div class="icp-card">
                    <div class="icp-card-header">
                        <h3>Total Space Saved</h3>
                    </div>
                    <div class="icp-card-body">
                        <div class="icp-stat-number"><?php echo $this->format_bytes($total_savings->total_savings); ?></div>
                    </div>
                </div>
                
                <div class="icp-card">
                    <div class="icp-card-header">
                        <h3>Average Savings</h3>
                    </div>
                    <div class="icp-card-body">
                        <div class="icp-stat-number">
                            <?php 
                            $avg_savings = $total_savings->total_original_size > 0 ? 
                                (($total_savings->total_savings / $total_savings->total_original_size) * 100) : 0;
                            echo round($avg_savings, 1) . '%';
                            ?>
                        </div>
                    </div>
                </div>
                
                <div class="icp-card">
                    <div class="icp-card-header">
                        <h3>Original Total Size</h3>
                    </div>
                    <div class="icp-card-body">
                        <div class="icp-stat-number"><?php echo $this->format_bytes($total_savings->total_original_size); ?></div>
                    </div>
                </div>
            </div>
            
            <!-- Recent Compressions Table -->
            <div class="icp-recent-compressions">
                <h2>Recent Compressions</h2>
                
                <!-- Search and Filter Form -->
                <div class="icp-table-controls">
                    <form method="get" class="icp-search-form">
                        <input type="hidden" name="page" value="image-compression" />
                        
                        <div class="icp-search-filters">
                            <div class="icp-filter-group">
                                <label for="search">Search:</label>
                                <input type="text" id="search" name="search" value="<?php echo esc_attr($search); ?>" placeholder="Search by image name or ID..." />
                            </div>
                            
                            <div class="icp-filter-group">
                                <label for="status">Status:</label>
                                <select id="status" name="status">
                                    <option value="">All Statuses</option>
                                    <?php foreach ($available_statuses as $status): ?>
                                        <option value="<?php echo esc_attr($status); ?>" <?php selected($status_filter, $status); ?>>
                                            <?php echo ucfirst(esc_html($status)); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="icp-filter-group">
                                <label for="date_from">From:</label>
                                <input type="date" id="date_from" name="date_from" value="<?php echo esc_attr($date_from); ?>" />
                            </div>
                            
                            <div class="icp-filter-group">
                                <label for="date_to">To:</label>
                                <input type="date" id="date_to" name="date_to" value="<?php echo esc_attr($date_to); ?>" />
                            </div>
                            
                            <div class="icp-filter-group">
                                <label for="per_page">Show:</label>
                                <select id="per_page" name="per_page">
                                    <option value="10" <?php selected($per_page, 10); ?>>10 per page</option>
                                    <option value="25" <?php selected($per_page, 25); ?>>25 per page</option>
                                    <option value="50" <?php selected($per_page, 50); ?>>50 per page</option>
                                </select>
                            </div>
                            
                            <div class="icp-filter-actions">
                                <input type="submit" class="button" value="Filter" />
                                <a href="<?php echo admin_url('admin.php?page=image-compression'); ?>" class="button button-secondary">Clear</a>
                                <div class="icp-search-hint">Press <kbd>Ctrl</kbd>+<kbd>K</kbd> to focus search</div>
                            </div>
                        </div>
                    </form>
                </div>
                
                <?php if (empty($recent_compressions)): ?>
                    <p>No images found matching your criteria. <a href="<?php echo admin_url('admin.php?page=image-compression-bulk'); ?>">Start bulk compression</a> or try adjusting your search filters.</p>
                <?php else: ?>
                    <table class="wp-list-table widefat fixed striped icp-results-table">
                        <thead>
                            <tr>
                                <th>Image ID</th>
                                <th>Image Name</th>
                                <th>Original Size</th>
                                <th>Compressed Size</th>
                                <th>Savings</th>
                                <th>Date</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_compressions as $compression): ?>
                                <tr>
                                    <td><?php echo $compression->attachment_id; ?></td>
                                    <td><?php echo $compression->image_name ? esc_html($compression->image_name) : get_the_title($compression->attachment_id); ?></td>
                                    <td><?php echo $this->format_bytes($compression->original_size); ?></td>
                                    <td><?php echo $this->format_bytes($compression->compressed_size); ?></td>
                                    <td>
                                        <span class="icp-savings">
                                            <?php echo round($compression->savings_percent, 1); ?>% 
                                            (<?php echo $this->format_bytes($compression->original_size - $compression->compressed_size); ?>)
                                        </span>
                                    </td>
                                    <td><?php echo date('M j, Y g:i A', strtotime($compression->compression_date)); ?></td>
                                    <td>
                                        <?php if ($compression->status === 'success'): ?>
                                            <?php if ($compression->is_restored): ?>
                                                <span class="icp-status-restored">↻ Restored</span>
                                            <?php else: ?>
                                                <span class="icp-status-success">✓ Success</span>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span class="icp-status-error" title="<?php echo esc_attr($compression->error_message); ?>">✗ Failed</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    
                    <!-- Pagination Controls -->
                    <?php if ($compression_data['total_pages'] > 1): ?>
                        <div class="icp-pagination">
                            <div class="icp-pagination-info">
                                Showing <?php echo (($current_page - 1) * $per_page + 1); ?> to 
                                <?php echo min($current_page * $per_page, $compression_data['total_items']); ?> of 
                                <?php echo $compression_data['total_items']; ?> results
                            </div>
                            
                            <div class="icp-pagination-links">
                                <?php
                                $base_url = admin_url('admin.php?page=image-compression');
                                $query_params = array();
                                if ($search) $query_params['search'] = $search;
                                if ($status_filter) $query_params['status'] = $status_filter;
                                if ($date_from) $query_params['date_from'] = $date_from;
                                if ($date_to) $query_params['date_to'] = $date_to;
                                if ($per_page != 10) $query_params['per_page'] = $per_page;
                                
                                // Previous page
                                if ($current_page > 1): 
                                    $prev_params = array_merge($query_params, array('paged' => $current_page - 1));
                                    $prev_url = add_query_arg($prev_params, $base_url);
                                ?>
                                    <a href="<?php echo esc_url($prev_url); ?>" class="button">&laquo; Previous</a>
                                <?php endif; ?>
                                
                                <?php
                                // Page numbers
                                $start_page = max(1, $current_page - 2);
                                $end_page = min($compression_data['total_pages'], $current_page + 2);
                                
                                for ($i = $start_page; $i <= $end_page; $i++):
                                    if ($i == $current_page):
                                ?>
                                    <span class="button button-primary"><?php echo $i; ?></span>
                                <?php else:
                                    $page_params = array_merge($query_params, array('paged' => $i));
                                    $page_url = add_query_arg($page_params, $base_url);
                                ?>
                                    <a href="<?php echo esc_url($page_url); ?>" class="button"><?php echo $i; ?></a>
                                <?php endif; endfor; ?>
                                
                                <?php
                                // Next page
                                if ($current_page < $compression_data['total_pages']): 
                                    $next_params = array_merge($query_params, array('paged' => $current_page + 1));
                                    $next_url = add_query_arg($next_params, $base_url);
                                ?>
                                    <a href="<?php echo esc_url($next_url); ?>" class="button">Next &raquo;</a>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    public function bulk_compression_page() {
        // Get bulk compression settings
        $bulk_settings = $this->get_bulk_compression_settings();
        $excluded_folders = isset($bulk_settings['excluded_folders']) ? $bulk_settings['excluded_folders'] : array();

        $uncompressed_images = ICP_Database::get_uncompressed_images($excluded_folders);
        $failed_images = ICP_Database::get_failed_compression_images($excluded_folders);
        $total_uncompressed = count($uncompressed_images);
        $total_failed = count($failed_images);
        
        // Count truly uncompressed (never tried) vs failed images
        $never_compressed = 0;
        $failed_retry = 0;
        foreach ($uncompressed_images as $image) {
            if (empty($image->compression_status)) {
                // Never been compressed
                $never_compressed++;
            } else {
                // Check if this is a restored image (should be treated as never compressed)
                $compression_stats = ICP_Database::get_compression_stats($image->ID);
                if ($compression_stats && $compression_stats->is_restored == 1) {
                    // Restored images should be treated as never compressed for bulk compression UI
                    $never_compressed++;
                } else {
                    // Truly failed compression attempts
                    $failed_retry++;
                }
            }
        }
        
        // Check if API key is configured
        $settings = get_option('icp_settings');
        $api_key = isset($settings['api_key']) ? trim($settings['api_key']) : '';
        $api_key_configured = !empty($api_key);
        
        ?>
        <div class="wrap">
            <h1>Bulk Image Compression</h1>
            
            <div class="icp-bulk-compression">
                <?php if (!$api_key_configured): ?>
                    <div class="notice notice-warning">
                        <p><strong>🔑 TinyPNG API Key Required</strong></p>
                        <p>You need to configure your TinyPNG API key before you can use bulk compression.</p>
                        <p>
                            <a href="<?php echo admin_url('admin.php?page=image-compression-settings'); ?>" class="button button-primary">
                                Configure API Key
                            </a>
                            <a href="https://tinypng.com/developers" target="_blank" class="button button-secondary">
                                Get Free API Key
                            </a>
                        </p>
                    </div>
                <?php elseif ($total_uncompressed > 0): ?>
                    <div class="icp-bulk-info">
                        <h3>📊 Compression Status Overview</h3>
                        <p>Found <strong><?php echo number_format($total_uncompressed); ?></strong> images that need compression:</p>
                        <ul style="list-style: disc; margin-left: 20px;">
                            <?php if ($never_compressed > 0): ?>
                                <li><strong><?php echo number_format($never_compressed); ?></strong> images have never been compressed</li>
                            <?php endif; ?>
                            <?php if ($failed_retry > 0): ?>
                                <li><strong><?php echo number_format($failed_retry); ?></strong> images failed previous compression attempts and will be retried</li>
                            <?php endif; ?>
                        </ul>
                        <p>Click the button below to start compressing all images. This process may take some time depending on the number of images.</p>
                        
                        <!-- Show failed images summary if any exist -->
                        <?php if ($failed_retry > 0): ?>
                            <div class="notice notice-warning" style="margin-top: 15px;">
                                <p><strong>⚠️ Previous Failed Compressions</strong></p>
                                <p>Some images failed to compress in previous attempts. These will be automatically included in the bulk compression and retried. Common reasons for failures include:</p>
                                <ul style="list-style: disc; margin-left: 20px;">
                                    <li>Network connectivity issues</li>
                                    <li>TinyPNG API rate limits</li>
                                    <li>Corrupted or unsupported image files</li>
                                    <li>File permission issues</li>
                                </ul>
                                <p><strong>Tip:</strong> You can view detailed error messages in the <a href="<?php echo admin_url('admin.php?page=image-compression'); ?>">Dashboard</a> by filtering for "failed" status.</p>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="icp-bulk-controls">
                        <button id="icp-start-bulk-compression" class="button button-primary button-large">
                            Start Bulk Compression (<?php echo number_format($total_uncompressed); ?> images)
                        </button>
                        <?php if ($failed_retry > 0): ?>
                            <button id="icp-retry-failed-only" class="button button-secondary button-large" style="margin-left: 10px;">
                                Retry Failed Only (<?php echo number_format($failed_retry); ?> images)
                            </button>
                        <?php endif; ?>
                        <button id="icp-stop-bulk-compression" class="button button-secondary" style="display: none;">
                            Stop Compression
                        </button>
                    </div>

                    <!-- Folder Exclusion Settings for Bulk Compression -->
                    <?php
                    // Check if FileBird is available and show folder exclusion options
                    if (ICP_Database::is_filebird_active()):
                        $folder_hierarchy = ICP_Database::get_filebird_folder_hierarchy();
                        $bulk_excluded_folders = isset($bulk_settings['excluded_folders']) ? $bulk_settings['excluded_folders'] : array();
                    ?>

                    <div class="icp-bulk-folder-exclusion" style="margin-top: 20px; padding: 15px; border: 1px solid #ddd; border-radius: 4px; background: #f9f9f9;">
                        <h3>📁 Folder Exclusion Settings</h3>
                        <p class="description">Since FileBird plugin is detected, you can exclude specific folders from bulk compression. Images in excluded folders (including subfolders) will not be processed during bulk compression.</p>

                        <form id="icp-bulk-folder-exclusion-form">
                            <?php wp_nonce_field('icp_bulk_settings_nonce', 'bulk_settings_nonce'); ?>

                            <?php if (!empty($folder_hierarchy)): ?>
                                <div class="icp-folder-exclusion-container">
                                    <p class="description" style="margin-bottom: 10px;">
                                        <strong>Select folders to exclude from bulk compression:</strong><br>
                                        Selecting a parent folder will automatically exclude all its subfolders.
                                    </p>

                                    <div class="icp-folder-tree">
                                        <?php $this->render_bulk_folder_tree($folder_hierarchy, $bulk_excluded_folders); ?>
                                    </div>

                                    <p class="description" style="margin-top: 10px;">
                                        <em>Note: Manual compression will still work on images in excluded folders.</em>
                                    </p>

                                    <div style="margin-top: 15px;">
                                        <button type="button" id="icp-save-bulk-folder-settings" class="button button-secondary">
                                            Save Folder Exclusion Settings
                                        </button>
                                        <span id="icp-bulk-folder-settings-status" style="margin-left: 10px;"></span>
                                    </div>
                                </div>
                            <?php else: ?>
                                <p class="description">No FileBird folders found. Create some folders in your Media Library first.</p>
                            <?php endif; ?>
                        </form>
                    </div>

                    <?php else: ?>

                    <div class="notice notice-info inline" style="margin-top: 20px;">
                        <p>
                            <strong>Folder Exclusion:</strong>
                            Install and activate the <a href="https://wordpress.org/plugins/filebird/" target="_blank">FileBird plugin</a>
                            to enable folder-based exclusion for bulk compression. This allows you to organize your media
                            library into folders and exclude specific folders from bulk compression.
                        </p>
                    </div>

                    <?php endif; ?>

                    <div id="icp-bulk-progress" style="display: none;">
                        <div class="icp-progress-bar">
                            <div class="icp-progress-fill" style="width: 0%;"></div>
                        </div>
                        <div class="icp-progress-text">
                            <span id="icp-progress-current">0</span> / <span id="icp-progress-total"><?php echo $total_uncompressed; ?></span> images processed
                        </div>
                    </div>
                    
                    <div id="icp-bulk-results" style="display: none;">
                        <h3>Compression Results</h3>
                        <div id="icp-results-content"></div>
                    </div>
                    
                <?php else: ?>
                    <div class="notice notice-info">
                        <p>All images in your media library have already been compressed! 🎉</p>
                        <p><a href="<?php echo admin_url('admin.php?page=image-compression'); ?>">View compression statistics</a></p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    public function settings_page() {
        $settings = get_option('icp_settings');
        
        ?>
        <div class="wrap">
            <h1>Image Compression Settings</h1>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('icp_settings_group');
                do_settings_sections('icp_settings_group');
                ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">TinyPNG API Key</th>
                        <td>
                            <input type="text" name="icp_settings[api_key]" value="<?php echo esc_attr($settings['api_key']); ?>" class="regular-text" />
                            <p class="description">
                                Enter your TinyPNG API key. Get one free at <a href="https://tinypng.com/developers" target="_blank">tinypng.com/developers</a>
                            </p>
                            <?php if (!empty($settings['api_key'])): ?>
                                <button type="button" id="icp-test-api-key" class="button button-secondary">Test API Key</button>
                                <span id="icp-api-test-result"></span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Auto Compression</th>
                        <td>
                            <label>
                                <input type="checkbox" name="icp_settings[auto_compress]" value="1" <?php checked($settings['auto_compress'], 1); ?> />
                                Automatically compress images when uploaded
                            </label>
                            <p class="description">When enabled, all newly uploaded images will be automatically compressed.</p>
                        </td>
                    </tr>
                    

                </table>
                
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
    
    public function scheduler_page() {
        // Get scheduler instance to access status
        static $scheduler = null;
        if ($scheduler === null) {
            $scheduler = new ICP_Scheduler();
        }
        
        $status = $scheduler->get_schedule_status();
        $logs = $scheduler->get_scheduler_logs(10);
        
        ?>
        <div class="wrap">
            <h1>Scheduled Compression</h1>
            
            <div class="icp-scheduler-container">
                <!-- Scheduler Settings -->
                <div class="icp-scheduler-settings">
                    <h2>Scheduler Settings</h2>
                    
                    <form id="icp-scheduler-form">
                        <table class="form-table">
                            <tr>
                                <th scope="row">Enable Scheduled Compression</th>
                                <td>
                                    <label>
                                        <input type="checkbox" id="icp-scheduler-enabled" <?php checked($status['enabled'], true); ?> />
                                        Automatically compress uncompressed images
                                    </label>
                                    <p class="description">When enabled, the system will automatically check for and compress uncompressed images based on the schedule below.</p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">Compression Schedule</th>
                                <td>
                                    <select id="icp-scheduler-interval">
                                        <option value="every_minute" <?php selected($status['interval'], 'every_minute'); ?>>Every Minute (Testing)</option>
                                        <option value="every_5_minutes" <?php selected($status['interval'], 'every_5_minutes'); ?>>Every 5 Minutes</option>
                                        <option value="every_15_minutes" <?php selected($status['interval'], 'every_15_minutes'); ?>>Every 15 Minutes</option>
                                        <option value="every_30_minutes" <?php selected($status['interval'], 'every_30_minutes'); ?>>Every 30 Minutes</option>
                                        <option value="hourly" <?php selected($status['interval'], 'hourly'); ?>>Every Hour</option>
                                        <option value="twicedaily" <?php selected($status['interval'], 'twicedaily'); ?>>Twice Daily</option>
                                        <option value="daily" <?php selected($status['interval'], 'daily'); ?>>Daily</option>
                                        <option value="weekly" <?php selected($status['interval'], 'weekly'); ?>>Weekly</option>
                                    </select>
                                    <p class="description">How often should the system check for uncompressed images? For production sites, hourly or daily is recommended.</p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">Batch Processing</th>
                                <td>
                                    <label>
                                        <input type="radio" name="icp-batch-mode" id="icp-batch-unlimited" value="unlimited" <?php checked($status['batch_size'], -1); ?> />
                                        <strong>Compress all uncompressed images at once (No limit)</strong>
                                    </label>
                                    <p class="description" style="margin: 8px 0 15px 25px; color: #d63638;">
                                        ⚠️ <strong>Warning:</strong> This will process ALL uncompressed images in a single run. Use only if your server can handle large batches without timing out.
                                    </p>
                                    
                                    <label>
                                        <input type="radio" name="icp-batch-mode" id="icp-batch-limited" value="limited" <?php checked($status['batch_size'] != -1, true); ?> />
                                        <strong>Set custom batch size:</strong>
                                    </label>
                                    <div id="icp-batch-size-container" style="margin: 8px 0 0 25px; <?php echo ($status['batch_size'] == -1) ? 'display: none;' : ''; ?>">
                                        <input type="number" id="icp-scheduler-batch-size" value="<?php echo $status['batch_size'] == -1 ? 10 : esc_attr($status['batch_size']); ?>" min="1" style="width: 80px;" />
                                        <span> images per run</span>
                                        <p class="description" style="margin: 5px 0 0 0;">
                                            Enter any number of images to process per scheduled run. Lower numbers reduce server load but take more runs to complete all images.
                                        </p>
                                    </div>
                                </td>
                            </tr>
                        </table>
                        
                        <?php 
                        // Check if FileBird is available and show folder exclusion options
                        if (ICP_Database::is_filebird_active()): 
                            $folder_hierarchy = ICP_Database::get_filebird_folder_hierarchy();
                            $excluded_folders = isset($status['excluded_folders']) ? $status['excluded_folders'] : array();
                        ?>
                        
                        <h3>Folder Exclusion Settings</h3>
                        <p class="description">Since FileBird plugin is detected, you can exclude specific folders from scheduled compression. Images in excluded folders (including subfolders) will not be processed by the scheduler.</p>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">Exclude Folders</th>
                                <td>
                                    <?php if (!empty($folder_hierarchy)): ?>
                                        <div class="icp-folder-exclusion-container">
                                            <p class="description" style="margin-bottom: 10px;">
                                                <strong>Select folders to exclude from scheduled compression:</strong><br>
                                                Selecting a parent folder will automatically exclude all its subfolders.
                                            </p>
                                            
                                            <div class="icp-folder-tree">
                                                <?php $this->render_folder_tree($folder_hierarchy, $excluded_folders); ?>
                                            </div>
                                            
                                            <p class="description" style="margin-top: 10px;">
                                                <em>Note: Manual compression will still work on images in excluded folders.</em>
                                            </p>
                                        </div>
                                    <?php else: ?>
                                        <p class="description">No FileBird folders found. Create some folders in your Media Library first.</p>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        </table>
                        
                        <?php else: ?>
                        
                        <div class="notice notice-info inline">
                            <p>
                                <strong>Folder Exclusion:</strong> 
                                Install and activate the <a href="https://wordpress.org/plugins/filebird/" target="_blank">FileBird plugin</a> 
                                to enable folder-based exclusion for scheduled compression. This allows you to organize your media 
                                library into folders and exclude specific folders from automatic compression.
                            </p>
                        </div>
                        
                        <?php endif; ?>
                        
                        <div class="icp-scheduler-actions">
                            <button type="button" id="icp-update-scheduler" class="button button-primary">Save Settings</button>
                            <button type="button" id="icp-run-scheduler-now" class="button button-secondary">Run Now</button>
                            <span id="icp-scheduler-status"></span>
                        </div>
                    </form>
                </div>
                
                <!-- Scheduler Status -->
                <div class="icp-scheduler-status">
                    <h2>Current Status</h2>
                    
                    <div class="icp-status-cards">
                        <div class="icp-status-card">
                            <h3>Scheduler Status</h3>
                            <div class="icp-status-value">
                                <span class="icp-status-indicator <?php echo $status['enabled'] ? 'enabled' : 'disabled'; ?>">
                                    <?php echo $status['enabled'] ? '● Active' : '○ Inactive'; ?>
                                </span>
                            </div>
                        </div>
                        
                        <div class="icp-status-card">
                            <h3>Next Run</h3>
                            <div class="icp-status-value">
                                <?php if ($status['next_run']): ?>
                                    <span class="icp-next-run"><?php echo esc_html($status['next_run']); ?></span>
                                <?php else: ?>
                                    <span class="icp-no-schedule">Not scheduled</span>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="icp-status-card">
                            <h3>Batch Size</h3>
                            <div class="icp-status-value">
                                <?php if ($status['batch_size'] == -1): ?>
                                    <span class="icp-unlimited-batch">Unlimited</span>
                                    <div class="icp-batch-info">
                                        <small>All uncompressed images per run</small>
                                    </div>
                                <?php else: ?>
                                    <span class="icp-batch-size"><?php echo esc_html($status['batch_size']); ?> images</span>
                                    <div class="icp-batch-info">
                                        <small>Per scheduled run</small>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="icp-status-card">
                            <h3>Last Run</h3>
                            <div class="icp-status-value">
                                <?php if ($status['last_run']): ?>
                                    <span class="icp-last-run"><?php echo esc_html($status['last_run']); ?></span>
                                    <?php if ($status['last_run_stats']): ?>
                                        <div class="icp-last-run-stats">
                                            <small>
                                                <?php echo $status['last_run_stats']['successful']; ?>/<?php echo $status['last_run_stats']['processed']; ?> images compressed
                                                <?php if ($status['last_run_stats']['errors'] > 0): ?>
                                                    | <?php echo $status['last_run_stats']['errors']; ?> errors
                                                <?php endif; ?>
                                            </small>
                                        </div>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <span class="icp-no-run">Never run</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Activity Logs -->
                <div class="icp-scheduler-logs">
                    <h2>Recent Activity</h2>
                    
                    <div class="icp-logs-container">
                        <?php if (!empty($logs)): ?>
                            <table class="widefat striped">
                                <thead>
                                    <tr>
                                        <th>Time</th>
                                        <th>Type</th>
                                        <th>Message</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($logs as $log): ?>
                                        <tr>
                                            <td><?php echo esc_html($log['timestamp']); ?></td>
                                            <td><span class="icp-log-type icp-log-<?php echo esc_attr($log['type']); ?>"><?php echo esc_html(ucfirst($log['type'])); ?></span></td>
                                            <td><?php echo esc_html($log['message']); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php else: ?>
                            <p>No activity logs yet.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function ajax_test_api_key() {
        check_ajax_referer('icp_test_api', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $api_key = sanitize_text_field($_POST['api_key']);
        $result = $this->tinypng_api->test_api_key($api_key);
        
        wp_send_json($result);
    }
    
    public function ajax_reset_stats() {
        check_ajax_referer('icp_reset_stats', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        ICP_Database::reset_stats();
        
        wp_send_json_success('Statistics reset successfully');
    }
    
    public function ajax_cleanup_orphaned() {
        check_ajax_referer('icp_cleanup_orphaned', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $deleted_count = ICP_Database::cleanup_orphaned_records();
        
        if ($deleted_count !== false) {
            wp_send_json_success(array(
                'message' => sprintf('Successfully cleaned up %d orphaned record%s', $deleted_count, $deleted_count === 1 ? '' : 's'),
                'deleted_count' => $deleted_count
            ));
        } else {
            wp_send_json_error(array('message' => 'Failed to clean up orphaned records'));
        }
    }

    /**
     * AJAX handler for saving bulk compression settings
     */
    public function ajax_save_bulk_settings() {
        check_ajax_referer('icp_bulk_settings_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        // Get excluded folders from POST data
        $excluded_folders = isset($_POST['bulk_excluded_folders']) ? $_POST['bulk_excluded_folders'] : array();

        // Update bulk compression settings
        $settings = $this->update_bulk_compression_settings(array(
            'excluded_folders' => $excluded_folders
        ));

        wp_send_json_success(array(
            'message' => 'Bulk compression folder exclusion settings saved successfully',
            'settings' => $settings
        ));
    }

    public function enqueue_admin_assets($hook) {
        // Enqueue on our plugin pages
        $should_enqueue = strpos($hook, 'image-compression') !== false;
        
        // Also enqueue on media library pages and upload pages
        if (!$should_enqueue) {
            $media_pages = array('upload.php', 'media-new.php', 'post.php', 'post-new.php', 'media-upload.php');
            $should_enqueue = in_array($hook, $media_pages);
        }
        
        // Always enqueue on admin pages for media modal support
        if (!$should_enqueue && is_admin()) {
            global $pagenow;
            if (in_array($pagenow, array('upload.php', 'media-new.php', 'post.php', 'post-new.php', 'admin.php'))) {
                $should_enqueue = true;
            }
        }
        
        if (!$should_enqueue) {
            return;
        }
        
        wp_enqueue_style('icp-admin-style', ICP_PLUGIN_URL . 'assets/admin-style.css', array(), ICP_VERSION);
        wp_enqueue_script('icp-admin-script', ICP_PLUGIN_URL . 'assets/admin-script.js', array('jquery'), ICP_VERSION, true);
        
        wp_localize_script('icp-admin-script', 'icp_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'bulk_compress_nonce' => wp_create_nonce('icp_bulk_compress'),
            'compress_single_nonce' => wp_create_nonce('icp_compress_single'),
            'restore_single_nonce' => wp_create_nonce('icp_restore_single'),
            'bulk_restore_nonce' => wp_create_nonce('icp_bulk_restore'),
            'test_api_nonce' => wp_create_nonce('icp_test_api'),
            'reset_stats_nonce' => wp_create_nonce('icp_reset_stats'),
            'cleanup_orphaned_nonce' => wp_create_nonce('icp_cleanup_orphaned'),
            'get_recent_compressions_nonce' => wp_create_nonce('icp_get_recent_compressions'),
            'scheduler_nonce' => wp_create_nonce('icp_scheduler_nonce'),
            'bulk_settings_nonce' => wp_create_nonce('icp_bulk_settings_nonce')
        ));
    }
    
    public function enqueue_global_media_assets($hook) {
        // Always enqueue on upload.php (media library) regardless of view
        if ($hook === 'upload.php') {
            wp_enqueue_style('icp-admin-style', ICP_PLUGIN_URL . 'assets/admin-style.css', array(), ICP_VERSION);
            wp_enqueue_script('icp-admin-script', ICP_PLUGIN_URL . 'assets/admin-script.js', array('jquery'), ICP_VERSION, true);
            
            wp_localize_script('icp-admin-script', 'icp_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'bulk_compress_nonce' => wp_create_nonce('icp_bulk_compress'),
                'compress_single_nonce' => wp_create_nonce('icp_compress_single'),
                'restore_single_nonce' => wp_create_nonce('icp_restore_single'),
                'bulk_restore_nonce' => wp_create_nonce('icp_bulk_restore'),
                'test_api_nonce' => wp_create_nonce('icp_test_api'),
                'reset_stats_nonce' => wp_create_nonce('icp_reset_stats'),
                'cleanup_orphaned_nonce' => wp_create_nonce('icp_cleanup_orphaned'),
                'get_recent_compressions_nonce' => wp_create_nonce('icp_get_recent_compressions'),
                'scheduler_nonce' => wp_create_nonce('icp_scheduler_nonce'),
                'bulk_settings_nonce' => wp_create_nonce('icp_bulk_settings_nonce')
            ));
        }
    }
    
    public function enqueue_media_scripts() {
        // Enqueue our scripts when media modal is loaded
        wp_enqueue_style('icp-admin-style', ICP_PLUGIN_URL . 'assets/admin-style.css', array(), ICP_VERSION);
        wp_enqueue_script('icp-admin-script', ICP_PLUGIN_URL . 'assets/admin-script.js', array('jquery'), ICP_VERSION, true);
        
        wp_localize_script('icp-admin-script', 'icp_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'bulk_compress_nonce' => wp_create_nonce('icp_bulk_compress'),
            'compress_single_nonce' => wp_create_nonce('icp_compress_single'),
            'restore_single_nonce' => wp_create_nonce('icp_restore_single'),
            'bulk_restore_nonce' => wp_create_nonce('icp_bulk_restore'),
            'test_api_nonce' => wp_create_nonce('icp_test_api'),
            'reset_stats_nonce' => wp_create_nonce('icp_reset_stats'),
            'cleanup_orphaned_nonce' => wp_create_nonce('icp_cleanup_orphaned'),
            'get_recent_compressions_nonce' => wp_create_nonce('icp_get_recent_compressions'),
            'scheduler_nonce' => wp_create_nonce('icp_scheduler_nonce'),
            'bulk_settings_nonce' => wp_create_nonce('icp_bulk_settings_nonce')
        ));
    }
    
    public function add_settings_link($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=image-compression-settings') . '">Settings</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    public function add_media_column($columns) {
        $columns['compression'] = 'Compression';
        return $columns;
    }
    
    public function display_media_column($column_name, $attachment_id) {
        if ($column_name === 'compression') {
            $compression_data = ICP_Database::get_compression_stats($attachment_id);

            if ($compression_data) {
                if ($compression_data->status === 'success') {
                    echo '<div class="icp-compression-details">';

                    if ($compression_data->is_restored) {
                        // For restored images, show compress option instead of restoration details
                        echo '<a href="#" class="icp-compress-single" data-attachment-id="' . $attachment_id . '">Compress</a>';
                    } else {
                        echo '<span class="icp-compressed">✓ ' . round($compression_data->savings_percent, 1) . '% saved</span>';

                        echo '<div class="icp-compression-info">';
                        echo '<span class="icp-size-info">Original: ' . $this->format_bytes($compression_data->original_size) . '</span>';
                        echo '<span class="icp-size-info">Compressed: ' . $this->format_bytes($compression_data->compressed_size) . '</span>';
                        echo '<span class="icp-size-info">Saved: ' . $this->format_bytes($compression_data->original_size - $compression_data->compressed_size) . '</span>';
                        echo '</div>';

                        // Show restore button if backup exists and not already restored
                        if (!empty($compression_data->backup_path) && ICP_Backup::backup_exists($compression_data->backup_path)) {
                            echo '<div class="icp-restore-actions">';
                            echo '<a href="#" class="icp-restore-single button button-secondary button-small" data-attachment-id="' . $attachment_id . '">Restore Original</a>';
                            echo '</div>';
                        }
                    }

                    echo '</div>';
                } else {
                    echo '<div class="icp-compression-failed-wrapper">';
                    echo '<span class="icp-compression-failed" title="' . esc_attr($compression_data->error_message) . '">✗ Failed</span><br>';
                    echo '<small class="icp-error-details" title="' . esc_attr($compression_data->error_message) . '">' . esc_html(substr($compression_data->error_message, 0, 50)) . (strlen($compression_data->error_message) > 50 ? '...' : '') . '</small><br>';
                    echo '<a href="#" class="icp-compress-single button button-secondary button-small" data-attachment-id="' . $attachment_id . '">Retry</a>';
                    echo '</div>';
                }
            } else {
                $file_path = get_attached_file($attachment_id);
                $mime_type = wp_check_filetype($file_path)['type'];

                if (in_array($mime_type, array('image/jpeg', 'image/png', 'image/gif', 'image/webp'))) {
                    echo '<a href="#" class="icp-compress-single" data-attachment-id="' . $attachment_id . '">Compress</a>';
                } else {
                    echo '<span class="icp-not-supported">Not supported</span>';
                }
            }
        }
    }
    
    public function add_compression_field_to_modal($fields, $post) {
        // Only show for supported image types
        $mime_type = get_post_mime_type($post);
        if (!in_array($mime_type, array('image/jpeg', 'image/png', 'image/gif', 'image/webp'))) {
            return $fields;
        }
        
        $compression_data = ICP_Database::get_compression_stats($post->ID);
        
        if ($compression_data) {
            if ($compression_data->status === 'success') {
                if ($compression_data->is_restored) {
                    // For restored images, show compress option instead of restoration details
                    $html = '<div class="icp-modal-compression-status">';
                    $html .= '<span class="icp-not-compressed">Image restored - ready to compress</span><br>';
                    $html .= '<button type="button" class="button button-primary icp-compress-single-modal" data-attachment-id="' . $post->ID . '">Compress Image</button>';
                    $html .= '<div class="icp-compression-progress" style="display: none;">';
                    $html .= '<span class="spinner is-active"></span> Compressing...';
                    $html .= '</div>';
                    $html .= '</div>';
                } else {
                    $html = '<div class="icp-modal-compression-status">';
                    $html .= '<span class="icp-compressed">✓ Compressed - ' . round($compression_data->savings_percent, 1) . '% saved</span><br>';
                    $html .= '<small>Original: ' . $this->format_bytes($compression_data->original_size) . ' | ';
                    $html .= 'Compressed: ' . $this->format_bytes($compression_data->compressed_size) . ' | ';
                    $html .= 'Saved: ' . $this->format_bytes($compression_data->original_size - $compression_data->compressed_size) . '</small><br>';

                    // Show restore button if backup exists and not already restored
                    if (!empty($compression_data->backup_path) && ICP_Backup::backup_exists($compression_data->backup_path)) {
                        $html .= '<button type="button" class="button button-secondary icp-restore-single" data-attachment-id="' . $post->ID . '" style="margin-top: 10px;">Restore Original</button>';
                    }

                    $html .= '</div>';
                }
            } else {
                $html = '<div class="icp-modal-compression-status">';
                $html .= '<span class="icp-status-error">✗ Compression failed</span><br>';
                $html .= '<small title="' . esc_attr($compression_data->error_message) . '">' . esc_html($compression_data->error_message) . '</small><br>';
                $html .= '<button type="button" class="button button-secondary icp-compress-single-modal" data-attachment-id="' . $post->ID . '">Retry Compression</button>';
                $html .= '</div>';
            }
        } else {
            $html = '<div class="icp-modal-compression-status">';
            $html .= '<span class="icp-not-compressed">Not compressed</span><br>';
            $html .= '<button type="button" class="button button-primary icp-compress-single-modal" data-attachment-id="' . $post->ID . '">Compress Image</button>';
            $html .= '<div class="icp-compression-progress" style="display: none;">';
            $html .= '<span class="spinner is-active"></span> Compressing...';
            $html .= '</div>';
            $html .= '</div>';
        }
        
        $fields['icp_compression'] = array(
            'label' => 'Image Compression',
            'input' => 'html',
            'html' => $html,
        );
        
        return $fields;
    }
    
    /**
     * Render folder tree with checkboxes for exclusion
     */
    private function render_folder_tree($folders, $excluded_folders, $level = 0) {
        $indent = str_repeat('&nbsp;&nbsp;&nbsp;&nbsp;', $level);
        
        foreach ($folders as $folder) {
            $is_checked = in_array($folder['id'], $excluded_folders);
            $folder_name = esc_html($folder['name']);
            $folder_id = intval($folder['id']);
            
            echo '<div class="icp-folder-item" style="margin-left: ' . ($level * 20) . 'px;">';
            echo '<label>';
            echo '<input type="checkbox" name="excluded_folders[]" value="' . $folder_id . '"' . checked($is_checked, true, false) . ' />';
            echo '<span class="folder-icon">📁</span> ' . $folder_name;
            echo '</label>';
            echo '</div>';
            
            // Render children recursively
            if (!empty($folder['children'])) {
                $this->render_folder_tree($folder['children'], $excluded_folders, $level + 1);
            }
        }
    }

    /**
     * Render folder tree with checkboxes for bulk compression exclusion
     */
    private function render_bulk_folder_tree($folders, $excluded_folders, $level = 0) {
        $indent = str_repeat('&nbsp;&nbsp;&nbsp;&nbsp;', $level);

        foreach ($folders as $folder) {
            $is_checked = in_array($folder['id'], $excluded_folders);
            $folder_name = esc_html($folder['name']);
            $folder_id = intval($folder['id']);

            echo '<div class="icp-folder-item" style="margin-left: ' . ($level * 20) . 'px;">';
            echo '<label>';
            echo '<input type="checkbox" name="bulk_excluded_folders[]" value="' . $folder_id . '"' . checked($is_checked, true, false) . ' />';
            echo '<span class="folder-icon">📁</span> ' . $folder_name;
            echo '</label>';
            echo '</div>';

            // Render children recursively
            if (!empty($folder['children'])) {
                $this->render_bulk_folder_tree($folder['children'], $excluded_folders, $level + 1);
            }
        }
    }
    
    private function format_bytes($bytes) {
        if ($bytes >= 1048576) {
            return round($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return round($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' bytes';
        }
    }

    /**
     * Get bulk compression settings
     */
    public function get_bulk_compression_settings() {
        $settings = get_option('icp_bulk_settings', array());

        return array(
            'excluded_folders' => isset($settings['excluded_folders']) ? $settings['excluded_folders'] : array()
        );
    }

    /**
     * Update bulk compression settings
     */
    public function update_bulk_compression_settings($new_settings) {
        $current_settings = get_option('icp_bulk_settings', array());
        $settings = wp_parse_args($new_settings, $current_settings);

        // Validate excluded folders
        if (isset($settings['excluded_folders'])) {
            if (is_array($settings['excluded_folders'])) {
                // Filter out invalid folder IDs and convert to integers
                $settings['excluded_folders'] = array_filter(
                    array_map('intval', $settings['excluded_folders']),
                    function($id) { return $id > 0; }
                );
            } else {
                $settings['excluded_folders'] = array();
            }
        } else {
            $settings['excluded_folders'] = array();
        }

        // Update settings
        update_option('icp_bulk_settings', $settings);

        return $settings;
    }

    public function restore_page() {
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $per_page = 20;
        $offset = ($current_page - 1) * $per_page;

        // Get restorable images
        $restorable_images = ICP_Database::get_restorable_images($per_page, $offset);
        $total_restorable = ICP_Database::get_restorable_images_count();
        $total_pages = ceil($total_restorable / $per_page);

        ?>
        <div class="wrap">
            <h1>Restore Original Images</h1>

            <div class="icp-restore-intro">
                <p>This page shows all compressed images that have backup copies of their original files. You can restore individual images or select multiple images for bulk restore.</p>
            </div>

            <?php if (empty($restorable_images)): ?>
                <div class="notice notice-info">
                    <p><strong>No restorable images found.</strong></p>
                    <p>Images compressed after installing this restore feature will appear here if they have backup files available.</p>
                </div>
            <?php else: ?>
                <form id="icp-bulk-restore-form" method="post">
                    <?php wp_nonce_field('icp_bulk_restore', 'icp_bulk_restore_nonce'); ?>

                    <div class="tablenav top">
                        <div class="alignleft actions bulkactions">
                            <select name="action" id="bulk-action-selector-top">
                                <option value="-1">Bulk Actions</option>
                                <option value="restore">Restore Selected</option>
                            </select>
                            <input type="submit" id="doaction" class="button action" value="Apply">
                        </div>

                        <div class="tablenav-pages">
                            <span class="displaying-num"><?php echo $total_restorable; ?> items</span>
                            <?php if ($total_pages > 1): ?>
                                <?php
                                $page_links = paginate_links(array(
                                    'base' => add_query_arg('paged', '%#%'),
                                    'format' => '',
                                    'prev_text' => '&laquo;',
                                    'next_text' => '&raquo;',
                                    'total' => $total_pages,
                                    'current' => $current_page
                                ));
                                echo $page_links;
                                ?>
                            <?php endif; ?>
                        </div>
                    </div>

                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <td class="manage-column column-cb check-column">
                                    <input type="checkbox" id="cb-select-all-1">
                                </td>
                                <th class="manage-column">Image</th>
                                <th class="manage-column">Title</th>
                                <th class="manage-column">Compression Info</th>
                                <th class="manage-column">Backup Status</th>
                                <th class="manage-column">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($restorable_images as $image): ?>
                                <tr>
                                    <th class="check-column">
                                        <input type="checkbox" name="attachment_ids[]" value="<?php echo $image->attachment_id; ?>">
                                    </th>
                                    <td>
                                        <?php
                                        $thumbnail = wp_get_attachment_image($image->attachment_id, array(60, 60), true);
                                        echo $thumbnail ? $thumbnail : '<div class="attachment-preview"><div class="thumbnail"><div class="centered">No preview</div></div></div>';
                                        ?>
                                    </td>
                                    <td>
                                        <strong><?php echo esc_html($image->post_title ?: 'Untitled'); ?></strong>
                                        <br><small><?php echo esc_html(basename($image->file_path)); ?></small>
                                    </td>
                                    <td>
                                        <div class="icp-compression-stats">
                                            <span class="icp-savings">Saved: <?php echo round($image->savings_percent, 1); ?>%</span><br>
                                            <small>
                                                Original: <?php echo $this->format_bytes($image->original_size); ?><br>
                                                Compressed: <?php echo $this->format_bytes($image->compressed_size); ?><br>
                                                Saved: <?php echo $this->format_bytes($image->original_size - $image->compressed_size); ?>
                                            </small>
                                        </div>
                                    </td>
                                    <td>
                                        <?php if ($image->is_restored): ?>
                                            <span class="icp-status-restored">✓ Restored</span><br>
                                            <small>Restored: <?php echo date('M j, Y g:i A', strtotime($image->restore_date)); ?></small>
                                        <?php elseif (ICP_Backup::backup_exists($image->backup_path)): ?>
                                            <span class="icp-status-available">✓ Backup Available</span><br>
                                            <small>Size: <?php echo $this->format_bytes(ICP_Backup::get_backup_size($image->backup_path)); ?></small>
                                        <?php else: ?>
                                            <span class="icp-status-missing">✗ Backup Missing</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if (!$image->is_restored && ICP_Backup::backup_exists($image->backup_path)): ?>
                                            <a href="#" class="icp-restore-single button button-secondary button-small"
                                               data-attachment-id="<?php echo $image->attachment_id; ?>">
                                                Restore Original
                                            </a>
                                        <?php elseif ($image->is_restored): ?>
                                            <span class="icp-already-restored">Already Restored</span>
                                        <?php else: ?>
                                            <span class="icp-cannot-restore">Cannot Restore</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>

                    <div class="tablenav bottom">
                        <div class="alignleft actions bulkactions">
                            <select name="action2" id="bulk-action-selector-bottom">
                                <option value="-1">Bulk Actions</option>
                                <option value="restore">Restore Selected</option>
                            </select>
                            <input type="submit" id="doaction2" class="button action" value="Apply">
                        </div>

                        <div class="tablenav-pages">
                            <?php if ($total_pages > 1): ?>
                                <?php echo $page_links; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </form>
            <?php endif; ?>
        </div>

        <style>
        .icp-compression-stats {
            font-size: 12px;
        }
        .icp-savings {
            color: #0073aa;
            font-weight: bold;
        }
        .icp-status-available {
            color: #46b450;
        }
        .icp-status-restored {
            color: #0073aa;
        }
        .icp-status-missing {
            color: #dc3232;
        }
        .icp-already-restored,
        .icp-cannot-restore {
            color: #666;
            font-style: italic;
        }
        .icp-restore-intro {
            background: #f1f1f1;
            padding: 15px;
            border-left: 4px solid #0073aa;
            margin: 20px 0;
        }
        </style>
        <?php
    }
}